/*

HASH ALGORITHM: lookup3
Parts copied and re-formatted from lookup3.c by Bob Jenkins
For the original documentation see below

-------------------------------------------------------------------------------
lookup3.c, by Bob Jenkins, May 2006, Public Domain.

These are functions for producing 32-bit hashes for hash table lookup.
hashword(), hashlittle(), hashlittle2(), hashbig(), mix(), and final()
are externally useful functions.  Routines to test the hash are included
if SELF_TEST is defined.  You can use this free for any purpose.  It's in
the public domain.  It has no warranty.

You probably want to use hashlittle().  hashlittle() and hashbig()
hash byte arrays.  hashlittle() is is faster than hashbig() on
little-endian machines.  Intel and AMD are little-endian machines.
On second thought, you probably want hashlittle2(), which is identical to
hashlittle() except it returns two 32-bit hashes for the price of one.
You could implement hashbig2() if you wanted but I haven't bothered here.

If you want to find a hash of, say, exactly 7 integers, do
  a = i1;  b = i2;  c = i3;
  mix(a,b,c);
  a += i4; b += i5; c += i6;
  mix(a,b,c);
  a += i7;
  final(a,b,c);
then use c as the hash value.  If you have a variable length array of
4-byte integers to hash, use hashword().  If you have a byte array (like
a character string), use hashlittle().  If you have several byte arrays, or
a mix of things, see the comments above hashlittle().

Why is this so big?  I read 12 bytes at a time into 3 4-byte integers,
then mix those integers.  This is fast (you can do a lot more thorough
mixing with 12*3 instructions on 3 integers than you can with 3 instructions
on 1 byte), but shoehorning those bytes into integers efficiently is messy.
-------------------------------------------------------------------------------
*/

#include "util/htab_hash.h"

#define HASH_LITTLE_ENDIAN 1

#define hashsize(n) ((uint32_t)1 << (n))
#define hashmask(n) (hashsize(n) - 1)
#define rot(x, k) (((x) << (k)) | ((x) >> (32 - (k))))

/*
-------------------------------------------------------------------------------
mix -- mix 3 32-bit values reversibly.

This is reversible, so any information in (a,b,c) before mix() is
still in (a,b,c) after mix().

If four pairs of (a,b,c) inputs are run through mix(), or through
mix() in reverse, there are at least 32 bits of the output that
are sometimes the same for one pair and different for another pair.
This was tested for:
* pairs that differed by one bit, by two bits, in any combination
  of top bits of (a,b,c), or in any combination of bottom bits of
  (a,b,c).
* "differ" is defined as +, -, ^, or ~^.  For + and -, I transformed
  the output delta to a Gray code (a^(a>>1)) so a string of 1's (as
  is commonly produced by subtraction) look like a single 1-bit
  difference.
* the base values were pseudorandom, all zero but one bit set, or
  all zero plus a counter that starts at zero.

Some k values for my "a-=c; a^=rot(c,k); c+=b;" arrangement that
satisfy this are
    4  6  8 16 19  4
    9 15  3 18 27 15
   14  9  3  7 17  3
Well, "9 15 3 18 27 15" didn't quite get 32 bits diffing
for "differ" defined as + with a one-bit base and a two-bit delta.  I
used http://burtleburtle.net/bob/hash/avalanche.html to choose
the operations, constants, and arrangements of the variables.

This does not achieve avalanche.  There are input bits of (a,b,c)
that fail to affect some output bits of (a,b,c), especially of a.  The
most thoroughly mixed value is c, but it doesn't really even achieve
avalanche in c.

This allows some parallelism.  Read-after-writes are good at doubling
the number of bits affected, so the goal of mixing pulls in the opposite
direction as the goal of parallelism.  I did what I could.  Rotates
seem to cost as much as shifts on every machine I could lay my hands
on, and rotates are much kinder to the top and bottom bits, so I used
rotates.
-------------------------------------------------------------------------------
*/
#define mix(a, b, c) \
{ \
	a -= c;  a ^= rot(c, 4);  c += b; \
	b -= a;  b ^= rot(a, 6);  a += c; \
	c -= b;  c ^= rot(b, 8);  b += a; \
	a -= c;  a ^= rot(c, 16);  c += b; \
	b -= a;  b ^= rot(a, 19);  a += c; \
	c -= b;  c ^= rot(b, 4);  b += a; \
}

/*
-------------------------------------------------------------------------------
final -- final mixing of 3 32-bit values (a,b,c) into c

Pairs of (a,b,c) values differing in only a few bits will usually
produce values of c that look totally different.  This was tested for
* pairs that differed by one bit, by two bits, in any combination
  of top bits of (a,b,c), or in any combination of bottom bits of
  (a,b,c).
* "differ" is defined as +, -, ^, or ~^.  For + and -, I transformed
  the output delta to a Gray code (a^(a>>1)) so a string of 1's (as
  is commonly produced by subtraction) look like a single 1-bit
  difference.
* the base values were pseudorandom, all zero but one bit set, or
  all zero plus a counter that starts at zero.

These constants passed:
 14 11 25 16 4 14 24
 12 14 25 16 4 14 24
and these came close:
  4  8 15 26 3 22 24
 10  8 15 26 3 22 24
 11  8 15 26 3 22 24
-------------------------------------------------------------------------------
*/
#define final(a, b, c) \
{ \
	c ^= b; c -= rot(b, 14); \
	a ^= c; a -= rot(c, 11); \
	b ^= a; b -= rot(a, 25); \
	c ^= b; c -= rot(b, 16); \
	a ^= c; a -= rot(c, 4);  \
	b ^= a; b -= rot(a, 14); \
	c ^= b; c -= rot(b, 24); \
}

/*
-------------------------------------------------------------------------------
hashlittle() -- hash a variable-length key into a 32-bit value
  k       : the key (the unaligned variable-length array of bytes)
  length  : the length of the key, counting by bytes
  initval : can be any 4-byte value
Returns a 32-bit value.  Every bit of the key affects every bit of
the return value.  Two keys differing by one or two bits will have
totally different hash values.

The best hash table sizes are powers of 2.  There is no need to do
mod a prime (mod is sooo slow!).  If you need less than 32 bits,
use a bitmask.  For example, if you need only 10 bits, do
  h = (h & hashmask(10));
In which case, the hash table should have hashsize(10) elements.

If you are hashing n strings (uint8_t **)k, do it like this:
  for (i=0, h=0; i<n; ++i) h = hashlittle( k[i], len[i], h);

By Bob Jenkins, 2006.  bob_jenkins@burtleburtle.net.  You may use this
code any way you wish, private, educational, or commercial.  It's free.

Use for hash table lookup, or anything where one collision in 2^^32 is
acceptable.  Do NOT use for cryptographic purposes.
-------------------------------------------------------------------------------
*/


#if defined(__GNUC__) && (__GNUC__ >= 7) && !defined(__clang__)
#define fallthrough_ok __attribute__ ((fallthrough))
#else
#define fallthrough_ok
#endif

// See below. This function intentionally accesses some bytes more.
__attribute__((no_sanitize_address))
#if defined(__clang__)
__attribute__((no_sanitize("unsigned-integer-overflow")))
#endif
uint32_t hashlittle(const void *key, size_t length, uint32_t initval)
{
	/* internal state */
	uint32_t a, b, c;
	union {
		const void *ptr;
		size_t i;
	} u;

	/* Set up the internal state */
	a = b = c = 0xdeadbeef + ((uint32_t)length) + initval;

	u.ptr = key;
	if (HASH_LITTLE_ENDIAN && ((u.i & 0x3) == 0)) {
		/* read 32-bit chunks */
		const uint32_t *k = (const uint32_t *)key;
		/* const uint8_t  *k8; */

		/* all but last block: */
		/* aligned reads and affect 32 bits of (a,b,c) */
		while (length > 12) {
			a += k[0];
			b += k[1];
			c += k[2];
			mix(a, b, c);
			length -= 12;
			k += 3;
		}

		/*
		 * handle the last (probably partial) block
		 *
		 * "k[2]&0xffffff" actually reads beyond the end of the string,
		 * but then masks off the part it's not allowed to read. Because
		 * the string is aligned, the masked-off tail is in the same
		 * word as the rest of the string.
		 * Every machine with memory protection I've seen does it on
		 * word boundaries, so is OK with this. But VALGRIND will still
		 * catch it and complain. The masking trick does make the hash
		 * noticably faster for short strings (like English words).
		 */
		switch (length) {
		case 12:
			c += k[2];
			b += k[1];
			a += k[0];
			break;
		case 11:
			c += k[2] & 0xffffff;
			b += k[1];
			a += k[0];
			break;
		case 10:
			c += k[2] & 0xffff;
			b += k[1];
			a += k[0];
			break;
		case 9:
			c += k[2] & 0xff;
			b += k[1];
			a += k[0];
			break;
		case 8:
			b += k[1];
			a += k[0];
			break;
		case 7:
			b += k[1] & 0xffffff;
			a += k[0];
			break;
		case 6:
			b += k[1] & 0xffff;
			a += k[0];
			break;
		case 5:
			b += k[1] & 0xff;
			a += k[0];
			break;
		case 4:
			a += k[0];
			break;
		case 3:
			a += k[0] & 0xffffff;
			break;
		case 2:
			a += k[0] & 0xffff;
			break;
		case 1:
			a += k[0] & 0xff;
			break;
		case 0:
			/* zero length strings require no mixing */
			return c;
		}

	} else if (HASH_LITTLE_ENDIAN && ((u.i & 0x1) == 0)) {
		/* read 16-bit chunks */
		const uint16_t *k = (const uint16_t *)key;
		const uint8_t  *k8;

		/* all but last block: aligned reads and different mixing */
		while (length > 12) {
			a += k[0] + (((uint32_t)k[1]) << 16);
			b += k[2] + (((uint32_t)k[3]) << 16);
			c += k[4] + (((uint32_t)k[5]) << 16);
			mix(a, b, c);
			length -= 12;
			k += 6;
		}

		/* handle the last (probably partial) block */
		k8 = (const uint8_t *)k;
		switch (length) {
		case 12:
			c += k[4] + (((uint32_t)k[5]) << 16);
			b += k[2] + (((uint32_t)k[3]) << 16);
			a += k[0] + (((uint32_t)k[1]) << 16);
			break;
		case 11:
			c += ((uint32_t)k8[10]) << 16;
			/* fall through */
		case 10:
			c += k[4];
			b += k[2] + (((uint32_t)k[3]) << 16);
			a += k[0] + (((uint32_t)k[1]) << 16);
			break;
		case 9:
			c += k8[8];
			/* fall through */
		case 8:
			b += k[2] + (((uint32_t)k[3]) << 16);
			a += k[0] + (((uint32_t)k[1]) << 16);
			break;
		case 7:
			b += ((uint32_t)k8[6]) << 16;
			/* fall through */
		case 6:
			b += k[2];
			a += k[0] + (((uint32_t)k[1]) << 16);
			break;
		case 5:
			b += k8[4];
			/* fall through */
		case 4:
			a += k[0] + (((uint32_t)k[1]) << 16);
			break;
		case 3:
			a += ((uint32_t)k8[2]) << 16;
			/* fall through */
		case 2:
			a += k[0];
			break;
		case 1:
			a += k8[0];
			break;
		case 0:
			return c;
		}

	} else {
		/* need to read the key one byte at a time */
		const uint8_t *k = (const uint8_t *)key;

		/* all but the last block: affect some 32 bits of (a, b, c) */
		while (length > 12) {
			a += k[0];
			a += ((uint32_t)k[1]) << 8;
			a += ((uint32_t)k[2]) << 16;
			a += ((uint32_t)k[3]) << 24;
			b += k[4];
			b += ((uint32_t)k[5]) << 8;
			b += ((uint32_t)k[6]) << 16;
			b += ((uint32_t)k[7]) << 24;
			c += k[8];
			c += ((uint32_t)k[9])  << 8;
			c += ((uint32_t)k[10]) << 16;
			c += ((uint32_t)k[11]) << 24;
			mix(a, b, c);
			length -= 12;
			k += 12;
		}

		/* last block: affect all 32 bits of (c) */
		/* all the case statements fall through */
		switch (length) {
		case 12:
			c += ((uint32_t)k[11]) << 24;
			fallthrough_ok;
		case 11:
			c += ((uint32_t)k[10]) << 16;
			fallthrough_ok;
		case 10:
			c += ((uint32_t)k[9]) << 8;
			fallthrough_ok;
		case 9:
			c += k[8];
			fallthrough_ok;
		case 8:
			b += ((uint32_t)k[7]) << 24;
			fallthrough_ok;
		case 7:
			b += ((uint32_t)k[6]) << 16;
			fallthrough_ok;
		case 6:
			b += ((uint32_t)k[5]) << 8;
			fallthrough_ok;
		case 5:
			b += k[4];
			fallthrough_ok;
		case 4:
			a += ((uint32_t)k[3]) << 24;
			fallthrough_ok;
		case 3:
			a += ((uint32_t)k[2]) << 16;
			fallthrough_ok;
		case 2:
			a += ((uint32_t)k[1]) << 8;
			fallthrough_ok;
		case 1:
			a += k[0];
			break;
		case 0:
			return c;
		}
	}

	final(a, b, c);
	return c;
}
