// SPDX-License-Identifier: BSD-3-Clause OR Apache-2.0
#ifndef BUNDLE_V7_BUNDLE7_H_INCLUDED
#define BUNDLE_V7_BUNDLE7_H_INCLUDED

#include "ud3tn/bundle.h"
#include "ud3tn/result.h"

#include <inttypes.h>
#include <stdbool.h>
#include <stddef.h>

// Default CRC type, see enum bundle_crc_type in bundle.h.
#ifndef DEFAULT_BPV7_CRC_TYPE
#define DEFAULT_BPV7_CRC_TYPE BUNDLE_CRC_TYPE_16
#endif // DEFAULT_CRC_TYPE

/**
 * Perform basic validation of the provided bundle.
 *
 * This function asserts basic properties that are specified as mandatory by RFC 9171 and are
 * not already asserted by the parser or bundle-creation functions provided in this component.
 */
bool bundle7_is_valid(const struct bundle *bundle);

/**
 * Returns the number of bytes that will be required for the CBORepresentation
 * of the passed unsigned integer.
 */
size_t bundle7_cbor_uint_sizeof(uint64_t num);

/**
 * Returns the number of bytes that will be required for the CBORepresentation
 * of the passed EID.
 */
size_t bundle7_eid_sizeof(const char *eid);

/**
 * Converts the unified uD3TN flags into BPv7-bis protocol-compliant block
 * processing flags.
 *
 * @return BPv7-bis bundle block processing flags
 */
uint16_t bundle7_convert_to_protocol_block_flags(
	const struct bundle_block *block);


/**
 * Returns the byte-length of the CBORepresentation of an extension block.
 */
size_t bundle7_block_get_size(struct bundle_block *block);

size_t bundle7_get_serialized_size(struct bundle *bundle);
size_t bundle7_get_serialized_size_without_payload(struct bundle *bundle);

/**
 * Recalculates the length of the primary block stored in the
 * "primary_block_length" field. You should call this function if you change
 * something in the primary block.
 */
void bundle7_recalculate_primary_block_length(struct bundle *bundle);

/**
 * Returns the minimal number of serialized bytes of the first fragment of the
 * given bundle.
 *
 * The minimal fragment contains:
 *
 *   - all extension blocks
 *   - minimal header for the payload block
 */
size_t bundle7_get_first_fragment_min_size(struct bundle *bundle);

/**
 * Returns the minimal number of serialized bytes of the last fragment of the
 * given bundle.
 *
 * The minimal fragment contains:
 *
 *   - all extension blocks containing the "replicated in every fragment" flag
 *   - minimal header for the payload block
 */
size_t bundle7_get_last_fragment_min_size(struct bundle *bundle);

#endif // BUNDLE_V7_BUNDLE7_H_INCLUDED
