# SPDX-License-Identifier: BSD-3-Clause OR Apache-2.0
import json
import jsonschema

import pytest

from ud3tn_utils.aap2 import (
    AAP2UnixClient,
    AuthType,
    BundleADU,
    BundleADUFlags,
    ResponseStatus,
)

from ud3tn_utils.config import (
    JSONConfigMessage,
    LegacyConfigMessage,
    make_contact,
    RouterCommand,
)

from .helpers import (
    AAP2_AGENT_ID,
    AAP2_SECRET,
    AAP2_SOCKET,
    TEST_AAP2,
    TEST_JSON_CONFIG,
    TEST_QUERY,
    UD3TN_CONFIG_EP,
)

NODE_EID = "dtn://receiver.dtn/"
NODE2_EID = "dtn://receiver2.dtn/"

CONTACT_1 = (10, 10, 1000)
CONTACT_2 = (20, 10, 2000)

CLA_STR = "smtcp:"

CONTACT_JSON_SCHEMA = {
    "type": "array",
    "items": {
        "type": "object",
        "properties": {
            "start": {"type": "integer", "minimum": 0},
            "end": {"type": "integer", "minimum": 0},
            "next_hop_node_id": {"type": "string", "minLength": 4},
            "next_hop_cla_addr": {"type": "string", "minLength": 1},
            "data_rate": {"type": "integer", "minimum": 0},
            "reachable_eids": {
                "type": "array",
                "items": {"type": "string", "minLength": 4},
            },
        },
        "additionalProperties": False,
        "required": ["start", "end", "data_rate"],
    }
}


def _send_config(rpc_client, cmd, eid, reachable_eids=None, contacts=None):
    MSG_TYPE = JSONConfigMessage if TEST_JSON_CONFIG else LegacyConfigMessage
    payload = bytes(MSG_TYPE(
        eid,
        CLA_STR,
        reachable_eids=reachable_eids,
        contacts=contacts,
        type=cmd,
    ))
    rpc_client.send_adu(
        BundleADU(
            dst_eid=UD3TN_CONFIG_EP,
            payload_length=len(payload),
            adu_flags=[BundleADUFlags.BUNDLE_ADU_WITH_BDM_AUTH],
        ),
        payload,
    )
    response = rpc_client.receive_response()
    assert response.response_status == ResponseStatus.RESPONSE_STATUS_SUCCESS


def _receive_response(sub_client):
    msg = sub_client.receive_msg()
    assert msg is not None
    assert msg.WhichOneof("msg") == "adu"
    adu_msg, bundle_data = sub_client.receive_adu(msg.adu)
    sub_client.send_response_status(
        ResponseStatus.RESPONSE_STATUS_SUCCESS
    )
    return json.loads(bundle_data)


@pytest.mark.skipif(not TEST_AAP2, reason="TEST_AAP2 disabled via environment")
@pytest.mark.skipif(not TEST_QUERY, reason="TEST_QUERY (BDM) not enabled")
def test_configure_query_delete_contact_aap2():
    eid = NODE_EID
    rpc_client = AAP2UnixClient(address=AAP2_SOCKET)
    sub_client = AAP2UnixClient(address=AAP2_SOCKET)
    with rpc_client, sub_client:
        print("Configuring the RPC client...")
        secret = rpc_client.configure(
            AAP2_AGENT_ID,
            subscribe=False,
            secret=AAP2_SECRET,
            auth_type=AuthType.AUTH_TYPE_BUNDLE_DISPATCH,
        )
        print("Configuring the subscriber client...")
        sub_client.configure(
            rpc_client.agent_id,
            subscribe=True,
            secret=secret,
        )
        contact1 = make_contact(*CONTACT_1)
        contact2 = make_contact(*CONTACT_2)
        print("Configuring 1st contact...")
        _send_config(
            rpc_client,
            RouterCommand.ADD,
            eid,
            contacts=[contact1],
        )
        _send_config(
            rpc_client,
            RouterCommand.QUERY,
            eid,
        )
        json_contact_data = _receive_response(sub_client)
        jsonschema.validate(json_contact_data, CONTACT_JSON_SCHEMA)
        assert len(json_contact_data) == 1
        assert json_contact_data[0]["start"] == contact1.start
        assert json_contact_data[0]["end"] == contact1.end
        assert json_contact_data[0]["next_hop_node_id"] == eid
        assert json_contact_data[0]["next_hop_cla_addr"] == CLA_STR
        assert json_contact_data[0]["data_rate"] == contact1.bitrate
        assert len(json_contact_data[0]["reachable_eids"]) == 1
        assert json_contact_data[0]["reachable_eids"][0] == eid
        _send_config(
            rpc_client,
            RouterCommand.ADD,
            eid,
            contacts=[contact2],
        )
        _send_config(
            rpc_client,
            RouterCommand.QUERY,
            eid,
        )
        json_contact_data = _receive_response(sub_client)
        jsonschema.validate(json_contact_data, CONTACT_JSON_SCHEMA)
        assert len(json_contact_data) == 2
        _send_config(
            rpc_client,
            RouterCommand.DELETE,
            eid,
            contacts=[contact1],
        )
        _send_config(
            rpc_client,
            RouterCommand.QUERY,
            eid,
        )
        json_contact_data = _receive_response(sub_client)
        jsonschema.validate(json_contact_data, CONTACT_JSON_SCHEMA)
        assert len(json_contact_data) == 1
        assert json_contact_data[0]["start"] == contact2.start
        assert json_contact_data[0]["end"] == contact2.end
        assert json_contact_data[0]["next_hop_node_id"] == eid
        assert json_contact_data[0]["next_hop_cla_addr"] == CLA_STR
        assert json_contact_data[0]["data_rate"] == contact2.bitrate
        assert len(json_contact_data[0]["reachable_eids"]) == 1
        assert json_contact_data[0]["reachable_eids"][0] == eid
        _send_config(
            rpc_client,
            RouterCommand.ADD,
            eid,
            reachable_eids=[NODE2_EID],
            contacts=[contact2],
        )
        _send_config(
            rpc_client,
            RouterCommand.QUERY,
            eid,
        )
        json_contact_data = _receive_response(sub_client)
        jsonschema.validate(json_contact_data, CONTACT_JSON_SCHEMA)
        assert len(json_contact_data) == 1
        assert json_contact_data[0]["start"] == contact2.start
        assert len(json_contact_data[0]["reachable_eids"]) == 2
        for reid in (eid, NODE2_EID):
            assert reid in json_contact_data[0]["reachable_eids"]
        _send_config(
            rpc_client,
            RouterCommand.DELETE,
            eid,
        )
        _send_config(
            rpc_client,
            RouterCommand.QUERY,
            eid,
        )
        json_contact_data = _receive_response(sub_client)
        jsonschema.validate(json_contact_data, CONTACT_JSON_SCHEMA)
        assert len(json_contact_data) == 0
